/*
 * released into the public domain
 * written by Mike Frysinger <vapier>
 */

#pragma GCC diagnostic warning "-Wall"
#pragma GCC diagnostic warning "-Wextra"

#define _FILE_OFFSET_BITS 64
#define _LARGEFILE_SOURCE
#define _LARGEFILE64_SOURCE
#define _GNU_SOURCE

#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <locale.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <time.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <linux/fs.h>

#define errp(msg, args...) \
	do { \
		printf("%s: " msg ": %m\n", program_invocation_short_name, ## args); \
		exit(1); \
	} while (0)

static unsigned long long mbps(struct timespec *stime, struct timespec *etime, off_t len)
{
	uint64_t dtime;
	clock_gettime(CLOCK_MONOTONIC, etime);
	dtime = ((etime->tv_sec - stime->tv_sec) * 1000 * 1000 * 1000) +
		(etime->tv_nsec - stime->tv_nsec);
	*stime = *etime;
	return 1000 * len / dtime;
}

static off_t get_blk_size(int fd)
{
	uint64_t size;
	if (ioctl(fd, BLKGETSIZE64, &size))
		errp("ioctl(BLKGETSIZE64) failed");
	return size;
}

static off_t get_size(int fd)
{
	struct stat st;

	if (fstat(fd, &st))
		errp("could not stat %i", fd);

	if (S_ISREG(st.st_mode))
		return st.st_size;
	else if (S_ISBLK(st.st_mode))
		return get_blk_size(fd);

	errno = EINVAL;
	errp("unknown type of file");
}

static void nuke(int fd, off_t offset, off_t max_size, unsigned char pattern)
{
	static char mem[1024 * 1024];
	memset(mem, pattern, sizeof(mem));
	if (pattern)
		printf("Writing 0x%X to the output\n", pattern);

	unsigned long long speed = 0;
	struct timespec stime, etime, itime;
	clock_gettime(CLOCK_MONOTONIC, &stime);
	itime = stime;

	off_t pos = offset, last_pos = 0;
	ssize_t ret;
	int fsync_pos = 0;
	while (pos < max_size) {
		/* This will round up to sizeof(mem) ... */
		ret = write(fd, mem, sizeof(mem));
		pos += ret;
		if (ret != sizeof(mem)) {
			printf("\nread() returned %zi (wanted %zu)\n%'llu MB/s over entire run\n",
				ret, sizeof(mem), mbps(&itime, &etime, pos));
			return;
		}

		printf("%'llu bytes %u%% (%'llu MB/s)%20s\r", (unsigned long long)pos,
			(unsigned)((pos * 100) / max_size), speed, "");

		if ((++fsync_pos % 16) == 0) {
			speed = mbps(&stime, &etime, pos - last_pos);
			last_pos = pos;
			fflush(stdout);
			fsync(fd);
		}
	}

	printf("\n");
}

static void usage(int status)
{
	fprintf(
		status ? stderr : stdout,
		"Usage: ddnuke [options] <dev>\n"
		"\n"
		"Options:"
		" -o <offset>   Position to start writing\n"
		" -r            Write random garbage to the device\n"
	);
	exit(status);
}

int main(int argc, char *argv[])
{
	off_t offset = 0;
	const char *file;
	bool random = false;
	int o;

	setlocale(LC_NUMERIC, "en_US");

	while ((o = getopt(argc, argv, "ho:r")) != -1) {
		switch (o) {
		case 'o':
			offset = atoll(optarg);
			break;
		case 'r':
			random = true;
			break;
		case 'h': usage(EX_OK);
		default:  usage(EX_USAGE);
		}
	}
	if (argc != optind + 1)
		usage(EX_USAGE);
	file = argv[optind];

	int fd = open(file, O_WRONLY|O_CLOEXEC);
	if (fd == -1)
		errp("open(%s) failed", file);

	off_t max_size = get_size(fd);

	if (lseek(fd, offset, SEEK_SET) != offset)
		errp("lseek(%"PRIu64"u) failed", offset);

	nuke(fd, offset, max_size, 0x00);
	if (random) {
		nuke(fd, offset, max_size, 0xaa);
		nuke(fd, offset, max_size, 0x55);
	}

	return EX_OK;
}
