/*
 * Copyright (C) 2010 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <errno.h>
#include <fcntl.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* This is from the kernel flash_ts.c driver */

typedef uint32_t u32;

/* Keep in sync with 'struct flash_ts' */
#define FLASH_TS_HDR_SIZE       (4 * sizeof(u32))
#define FLASH_TS_MAX_SIZE       (16 * 1024)
#define FLASH_TS_MAX_DATA_SIZE  (FLASH_TS_MAX_SIZE - FLASH_TS_HDR_SIZE)

#define FLASH_TS_MAGIC          0x53542a46

struct flash_ts {
  u32 magic;      /* "F*TS" */
  u32 crc;        /* doesn't include magic and crc fields */
  u32 len;        /* real size of data */
  u32 version;    /* generation counter, must be positive */

  /* data format is very similar to Unix environment:
   *   key1=value1\0key2=value2\0\0
   */
  char data[FLASH_TS_MAX_DATA_SIZE];
};

/* See if this sector is all 0xff (because it's been erased).
 * This is normal, so let's not be too whiny about it. */
static bool flash_is_blank(struct flash_ts *fts) {
  static struct flash_ts erased_fts;
  if (erased_fts.magic == 0)
    memset(&erased_fts, 0xff, sizeof(erased_fts));
  return !memcmp(&erased_fts, fts, sizeof(*fts));
}

/* Dump the key value array of data. */
static void fts_dump_data(const char data[static FLASH_TS_MAX_DATA_SIZE]) {
  const char *p = data;
  while (*p) {
    printf("%8s- %s\n", "", p);
    p += strlen(p) + 1;
  }
}

/* Dump the FTS of the open fd. */
static int fts_dump_fd(int fd) {
  off_t off = 0;
  size_t blank_cnt = 0;
  while (1) {
    struct flash_ts fts;
    ssize_t ret = read(fd, &fts, sizeof(fts));
    switch (ret) {
    case 0:
      return EXIT_SUCCESS;
    case sizeof(fts):
      break;
    default:
      perror("read error");
      return EXIT_FAILURE;
    }

    bool this_blank = flash_is_blank(&fts);
    if (this_blank) {
      switch (blank_cnt++) {
      case 0:
        break;
      case 1:
        printf("%8s\n", "*");
      default:
        goto done;
      }
    } else
      blank_cnt = 0;

    printf("%08llx: ", (unsigned long long)off);
    if (fts.magic == FLASH_TS_MAGIC) {
      printf("magic: %08x crc: %08x len: %8u ver: %8u\n",
             fts.magic, fts.crc, fts.len, fts.version);
      fts_dump_data(fts.data);
    } else {
      if (flash_is_blank(&fts))
        printf("erased\n");
      else
        printf("invalid magic\n");
    }

 done:
    off += ret;
  }
}

/* Dump the FTS specified by a file (normally /dev/mtd#). */
static int fts_dump(const char *file) {
  int fd = open(file, O_RDONLY|O_CLOEXEC);
  if (fd < 0)
    return EXIT_FAILURE;
  int ret = fts_dump_fd(fd);
  close(fd);
  return ret;
}

static void usage(int status) {
  fprintf(status ? stderr : stdout,
          "Usage: fts_dump /dev/mtd#\n");
  exit(status);
}

int main(int argc, char *argv[]) {
  if (argc != 2)
    usage(EXIT_FAILURE);
  if (!strcmp("-h", argv[1]))
    usage(EXIT_SUCCESS);

  return fts_dump(argv[1]);
}
