#define _GNU_SOURCE
#include <err.h>
#include <errno.h>
#include <getopt.h>
#include <inttypes.h>
#include <signal.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>

#define ARRAY_SIZE(a) (sizeof(a) / sizeof*(a))

static bool all;
static int verbose;

static void usage(int status)
{
	fprintf(status ? stderr : stdout,
		"Usage: sigmask-decode [options] <things to decode>\n"
		"\n"
		"The decodable objects:\n"
		"\t- a hex mask e.g. 0x123456\n"
		"\t- a pid e.g 12345\n"
		"\n"
		"Options:\n"
		" -a  Show all signals (otherwise only show ones in set)\n"
		" -v  More verbose output\n"
	);
	exit(status);
}

#define X(x) { #x, x },
static const struct {
	const char *name;
	int sig;
} signals[] = {
	X(SIGHUP)
	X(SIGINT)
	X(SIGQUIT)
	X(SIGILL)
	X(SIGTRAP)
	X(SIGABRT)
	X(SIGIOT)
	X(SIGBUS)
	X(SIGFPE)
	X(SIGKILL)
	X(SIGUSR1)
	X(SIGSEGV)
	X(SIGUSR2)
	X(SIGPIPE)
	X(SIGALRM)
	X(SIGTERM)
	X(SIGSTKFLT)
	X(SIGCHLD)
	X(SIGCONT)
	X(SIGSTOP)
	X(SIGTSTP)
	X(SIGTTIN)
	X(SIGTTOU)
	X(SIGURG)
	X(SIGXCPU)
	X(SIGXFSZ)
	X(SIGVTALRM)
	X(SIGPROF)
	X(SIGWINCH)
	X(SIGIO)
	X(SIGPWR)
	X(SIGSYS)
};

const char *strsig(int sig)
{
	static char rtsig[20];
	size_t i;

	for (i = 0; i < ARRAY_SIZE(signals); ++i)
		if (signals[i].sig == sig)
			return signals[i].name;

	/* We are decoding the signals as seen by the kernel, so don't
	 * use the C library's limits as it might reserve some for itself */
	if (sig >= __SIGRTMIN && sig <= __SIGRTMAX) {
		sprintf(rtsig, "SIGRT%i", sig);
		return rtsig;
	}

	return "SIG???";
}

static uint64_t xatoi(const char *s, int base)
{
	char *end;
	long long ret = strtoll(s, &end, base);
	if (*s && *end)
		errx(EX_DATAERR, "error: could not decode: %s", s);
	return ret;
}

static bool strstartswith(const char *s, const char *pfx)
{
	size_t l = strlen(pfx);
	return strncmp(s, pfx, l) == 0;
}

static void decode_mask(uint64_t mask)
{
	int i;

	sigset_t set;
	sigemptyset(&set);

	/* Decode the bits in the mask into the set */
	for (i = 0; i < 64; ++i)
		if (mask & (1ull << (i - 1)))
			sigaddset(&set, i);

	/* Now print out the mask */
	if (verbose == 0)
		printf("0x%016"PRIx64": ", mask);
	int sigpad = NSIG < 100 ? 2 : NSIG < 1000 ? 3 : 4;
	for (i = 1; i < NSIG; ++i) {
		const char *name = strsig(i);
		int ismember = sigismember(&set, i);

		if (ismember == -1)
			warn("%s", name);

		if (!all && ismember == 0)
			continue;

		if (verbose == 0) {
			if (!ismember)
				printf("!");
			printf("%s ", name);
		} else {
			printf("%*i: %-10s: %-3s", sigpad, i, name, ismember ? "IN" : "OUT");
			if (verbose > 1)
				printf(" (%s)", strsignal(i));
			printf("\n");
		}
	}
	if (verbose == 0)
		printf("\n");
}

static void decode_pid(int pid)
{
	static const struct {
		const char *pfx;
		const char *meaning;
	} lines[] = {
		{ "SigPnd", "Signals pending for this thread" },
		{ "ShdPnd", "Shared signals pending for this process" },
		{ "SigBlk", "Signals blocked" },
		{ "SigIgn", "Signals ignored" },
		{ "SigCgt", "Signals caught" },
	};

	char path[1024];
	char *line;
	size_t len, i;
	FILE *fp;

	sprintf(path, "/proc/%i/status", pid);
	fp = fopen(path, "r");
	if (!fp)
		err(EX_NOINPUT, "could not read: %s", path);

	line = NULL;
	while (getline(&line, &len, fp) != -1) {
		for (i = 0; i < ARRAY_SIZE(lines); ++i)
			if (strstartswith(line, lines[i].pfx)) {
				char *d = line;
				/* Skip the "SigPnd: " prefix */
				d += 8;
				/* Chop the trailing newline */
				if (d[16] != '\n')
					err(EX_SOFTWARE, "line {%s} not formatted as expected", line);
				d[16] = '\0';
				printf("%s: ", lines[i].pfx);
				decode_mask(xatoi(line + 8, 16));
			}
	}
	free(line);

	fclose(fp);
}

int main(int argc, char *argv[])
{
	int i;

	all = false;
	verbose = 0;
	while ((i = getopt(argc, argv, "ahv")) != -1) {
		switch (i) {
		case 'a': all = true; break;
		case 'v': ++verbose; break;

		case 'h': usage(EX_OK);
		default: usage(EX_USAGE);
		}
	}

	if (argc == optind)
		usage(EX_USAGE);

	for (i = optind; i < argc; ++i) {
		char *d = argv[i];

		if (strstartswith(d, "0x")) {
			decode_mask(xatoi(d, 16));
		} else {
			printf("PID %s:\n", d);
			decode_pid(xatoi(d, 10));
		}
	}

	return 0;
}
